<?php

###############################################################################
#
# Actions Factory Safe
#
# This action factory can be safely used in its firmware from b8 and higher.
# Incompatible actions will not cause fatal exceptions, the Dune HD application
# will quietly continue to work.
#
# Author: Brigadir (forum.mydune.ru)
# Date: 12-05-2019
# Latest update: 02-04-2023
#
###############################################################################

require_once 'lib/action_factory.php';

///////////////////////////////////////////////////////////////////////////////

class ActionFactorySafe extends ActionFactory
{

	// Folders ////////////////////////////////////////////////////////////////

	public static function open_folder($media_url_str = null, $caption = null, $id = null, $sel_id = null, $post_action = null, $keep_osd_context = false)
	{
		$action =
			array
			(
				GuiAction::handler_string_id => PLUGIN_OPEN_FOLDER_ACTION_ID,
				GuiAction::data =>
					array
					(
						PluginOpenFolderActionData::media_url => $media_url_str,
						PluginOpenFolderActionData::caption => $caption,
					)
			);

		if (!is_null($id) && defined('PluginOpenFolderActionData::id'))
			$action[GuiAction::data][PluginOpenFolderActionData::id] = $id;

		if (!is_null($sel_id) && defined('PluginOpenFolderActionData::sel_id'))
			$action[GuiAction::data][PluginOpenFolderActionData::sel_id] = $sel_id;

		if (!is_null($post_action) && defined('PluginOpenFolderActionData::post_action'))
			$action[GuiAction::data][PluginOpenFolderActionData::post_action] = $post_action;

		if ($keep_osd_context && defined('PluginOpenFolderActionData::keep_osd_context'))
			$action[GuiAction::data][PluginOpenFolderActionData::keep_osd_context] = $keep_osd_context;

		return $action;
	}

    public static function invalidate_folders($media_urls, $post_action = null, $all_except = false)
    {
        $action =
	        array
	        (
	            GuiAction::handler_string_id => PLUGIN_INVALIDATE_FOLDERS_ACTION_ID,
	            GuiAction::data =>
	                array
	                (
	                    PluginInvalidateFoldersActionData::media_urls => $media_urls,
	                    PluginInvalidateFoldersActionData::post_action => $post_action,
	                ),
	        );

        if ($all_except && defined('PluginInvalidateFoldersActionData::all_except'))
			$action[GuiAction::data][PluginInvalidateFoldersActionData::all_except] = true;

        return $action;
    }

    public static function invalidate_all_folders($post_action = null, $except_media_urls = null)
    {
    	if (!defined('PluginInvalidateFoldersActionData::all_except'))
    		return $post_action;

        return self::invalidate_folders($except_media_urls, $post_action, true);
    }

	public static function update_regular_folder($range, $need_refresh = false, $sel_ndx = -1)
	{
		return parent::update_regular_folder($range, $need_refresh, $sel_ndx);
	}

	public static function reset_controls($defs, $post_action = null, $initial_sel_ndx = -1)
	{
		return parent::reset_controls($defs, $post_action, $initial_sel_ndx);
	}

    ///////////////////////////////////////////////////////////////////////////


	// Dialogs ////////////////////////////////////////////////////////////////

	public static function show_error($fatal, $title, $msg_lines=null, $stop_playback=false, $retry_delay_ms=0)
	{
		if ($fatal)
			hd_print("Plugin fatal error!");

		$action =
			array
			(
				GuiAction::handler_string_id => PLUGIN_SHOW_ERROR_ACTION_ID,
				GuiAction::caption => null,
				GuiAction::data =>
					array
					(
						PluginShowErrorActionData::fatal => $fatal,
						PluginShowErrorActionData::title => $title,
						PluginShowErrorActionData::msg_lines => $msg_lines,
					),
				GuiAction::params => null,
			);

		if ($stop_playback && defined('PluginShowErrorActionData::stop_playback'))
			$action[GuiAction::data][PluginShowErrorActionData::stop_playback] = $stop_playback;

		if (!empty($retry_delay_ms) && defined('PluginShowErrorActionData::retry_delay_ms'))
			$action[GuiAction::data][PluginShowErrorActionData::retry_delay_ms] = $retry_delay_ms;

		return $action;
	}

	public static function show_dialog($title, $defs, $close_by_return = false, $preferred_width = 0, $attrs = array())
	{
		$action =
			array
			(
				GuiAction::handler_string_id => SHOW_DIALOG_ACTION_ID,
				GuiAction::caption => null,
				GuiAction::data =>
					array
					(
						ShowDialogActionData::title => $title,
						ShowDialogActionData::defs => $defs,
						ShowDialogActionData::close_by_return => $close_by_return,
						ShowDialogActionData::preferred_width => $preferred_width,
						ShowDialogActionData::min_item_title_width => isset($attrs['min_item_title_width'])? $attrs['min_item_title_width'] : 0,
						ShowDialogActionData::initial_sel_ndx => isset($attrs['initial_sel_ndx'])? $attrs['initial_sel_ndx'] : -1,
						ShowDialogActionData::actions => isset($attrs['actions'])? $attrs['actions'] : null,
						ShowDialogActionData::timer => isset($attrs['timer'])? $attrs['timer'] : null,
						ShowDialogActionData::params => isset($attrs['dialog_params'])? $attrs['dialog_params'] : array(),
					),
				GuiAction::params => null,
			);

		if (defined('ShowDialogActionData::max_height'))
		{
			$max_height = isset($attrs['max_height'])? $attrs['max_height'] : 0;

			if (!empty($max_height))
				$action[GuiAction::data][ShowDialogActionData::max_height] = $max_height;
		}

		return $action;
	}

	public static function show_title_dialog($title, $post_action = null)
	{
		$defs = array();

		ControlFactory::add_custom_close_dialog_and_apply_button($defs, null, 'OK', 300, $post_action);

		return
			self::show_dialog($title, $defs);
	}

	public static function close_dialog_and_run($post_action = null)
	{
		return
			parent::close_dialog_and_run($post_action);
	}

	public static function close_dialog()
	{
		return
			self::close_dialog_and_run(null);
	}

	public static function close_and_run($post_action = null)
	{
		return
			parent::close_and_run($post_action);
	}

	///////////////////////////////////////////////////////////////////////////


	// Popup menu /////////////////////////////////////////////////////////////

	public static function show_popup_menu($menu_items, $sel_ndx = 0)
	{
		return
			parent::show_popup_menu($menu_items, $sel_ndx);
	}

    public static function get_menu_item($caption, $action)
    {
        return
        	parent::get_menu_item($caption, $action);
    }

    ///////////////////////////////////////////////////////////////////////////


	// Playback ///////////////////////////////////////////////////////////////

    public static function launch_media_url_ext($url, $caption = null, $post_action = null, $play_in_background = false, $start_index = -1, $internal_player_only = false)
    {
        $action =
	        array
	        (
	             GuiAction::handler_string_id => LAUNCH_MEDIA_URL_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                LaunchMediaUrlActionData::url => $url,
		                LaunchMediaUrlActionData::post_action => $post_action,
		             ),
	        );

        if (!empty($caption) && defined('LaunchMediaUrlActionData::caption'))
			$action[GuiAction::data][LaunchMediaUrlActionData::caption] = $caption;

        if ($play_in_background && defined('LaunchMediaUrlActionData::play_in_background'))
			$action[GuiAction::data][LaunchMediaUrlActionData::play_in_background] = $play_in_background;

        if (($start_index > 0) && defined('LaunchMediaUrlActionData::start_index'))
			$action[GuiAction::data][LaunchMediaUrlActionData::start_index] = $start_index;

        if ($internal_player_only && defined('LaunchMediaUrlActionData::internal_player_only'))
			$action[GuiAction::data][LaunchMediaUrlActionData::internal_player_only] = $internal_player_only;

        return $action;
    }

    public static function launch_media_url($url, $post_action = null)
    {
        return
        	self::launch_media_url_ext($url, null, $post_action);
    }

    public static function launch_playlist($url, $start_index)
    {
        return
        	parent::launch_playlist($url, $start_index);
    }

    public static function launch_file($url, $caption = null) {

    	if (!defined('FILE_PLAY_ACTION_ID'))
    		return null;

    	$data =
    		array
    		(
    			FilePlayActionData::url => $url,
    		);

    	if (defined('FilePlayActionData::caption'))
            $data[FilePlayActionData::caption] = $caption;

        return
        	array
			(
				GuiAction::handler_string_id => FILE_PLAY_ACTION_ID,
				GuiAction::data => $data,
			);
    }

    // Launch TV playback
	public static function tv_play($media_url = null)
	{
		$action = array(GuiAction::handler_string_id => PLUGIN_TV_PLAY_ACTION_ID);

		if (is_null($media_url))
			return $action;

        if (is_string($media_url))
        {
            $action[GuiAction::params] = array('selected_media_url' => $media_url);
        }
        else if (is_object($media_url))
        {
			$action[GuiAction::data] =
				array
				(
					PluginTvPlayActionData::initial_group_id => isset($media_url->group_id)? $media_url->group_id : null,
					PluginTvPlayActionData::initial_channel_id => isset($media_url->channel_id)? $media_url->channel_id : null,
					PluginTvPlayActionData::initial_is_favorite => isset($media_url->is_favorite)? (boolean) $media_url->is_favorite : false,
					PluginTvPlayActionData::initial_archive_tm => isset($media_url->archive_tm)? (integer) $media_url->archive_tm : -1,
				);
		}

		return $action;
	}

	// Launch VOD playback
	public static function vod_play($vod_info = null)
	{
		return
			parent::vod_play($vod_info);
	}

    public static function stop_playback($wait_for_completion = false, $post_action = null)
    {
    	//if (!defined('STOP_PLAYBACK_ACTION_ID'))
    	//	return self::close_and_run($post_action);

		return
			array
			(
				 GuiAction::handler_string_id => STOP_PLAYBACK_ACTION_ID,
				 GuiAction::data =>
					 array
					 (
					    StopPlaybackActionData::wait_for_completion => $wait_for_completion,
					    StopPlaybackActionData::post_action => $post_action,
					 ),
			);
    }

    ///////////////////////////////////////////////////////////////////////////


    // TV mode ////////////////////////////////////////////////////////////////////

	public static function update_epg($channel_id, $clear, $day_start_tm_sec = 0, $programs = null, $post_action = null)
	{
		if (!defined('PLUGIN_UPDATE_EPG_ACTION_ID'))
			return $post_action;

		return
			array
			(
				 GuiAction::handler_string_id => PLUGIN_UPDATE_EPG_ACTION_ID,
				 GuiAction::data =>
					 array
					 (
						PluginUpdateEpgActionData::channel_id => $channel_id,
						PluginUpdateEpgActionData::clear => $clear,
						PluginUpdateEpgActionData::day_start_tm_sec => $day_start_tm_sec,
						PluginUpdateEpgActionData::programs => $programs,
						PluginUpdateEpgActionData::post_action => $post_action,
					 ),
			);
	}

    public static function update_dvb_channels($post_action = null)
    {
    	if (!defined('UPDATE_DVB_CHANNELS_ACTION_ID'))
    		return $post_action;

	    return
		    array
		    (
		         GuiAction::handler_string_id => UPDATE_DVB_CHANNELS_ACTION_ID,
		         GuiAction::data =>
			         array
			         (
			            UpdateDvbChannelsActionData::post_action => $post_action,
			         ),
		    );
    }

	///////////////////////////////////////////////////////////////////////////


    // User inputs ////////////////////////////////////////////////////////////

    public static function handle_user_input($params, $timeout_ms = null, $show_dialog_delay_ms = null)
    {
        $data = array();

        if (isset($timeout_ms) && defined('PluginHandleUserInputActionData::operation_timeout'))
            $data[PluginHandleUserInputActionData::operation_timeout] = intval($timeout_ms);

        if (isset($show_dialog_delay_ms) && defined('PluginHandleUserInputActionData::show_dialog_delay'))
            $data[PluginHandleUserInputActionData::show_dialog_delay] = intval($show_dialog_delay_ms);

        return
	        array
	        (
	            GuiAction::handler_string_id => PLUGIN_HANDLE_USER_INPUT_ACTION_ID,
	            GuiAction::caption => null,
	            GuiAction::data => empty($data)? null : $data,
	            GuiAction::params => $params,
	        );
    }

	public static function change_behaviour($actions, $timer = null, $post_action = null)
	{
		return
			parent::change_behaviour($actions, $timer, $post_action);
	}

    public static function timer($delay_ms, $reset_on_keypress = false, $clear_on_keypress = false)
    {
        $arr = array(GuiTimerDef::delay_ms => $delay_ms);

        if ($reset_on_keypress && defined('GuiTimerDef::reset_on_keypress'))
            $arr[GuiTimerDef::reset_on_keypress] = true;

        if ($clear_on_keypress && defined('GuiTimerDef::clear_on_keypress'))
            $arr[GuiTimerDef::clear_on_keypress] = true;

        return $arr;
    }

	public static function status($status)
	{
		return parent::status($status);
	}

	///////////////////////////////////////////////////////////////////////////


	// OSD ////////////////////////////////////////////////////////////////////

	public static function add_osd_image(&$comps, $image_url, $x, $y, $image_width = 0, $image_height = 0)
	{
		if (!class_exists('PluginOsdComponent'))
			return;

		$comps[] = array
		(
			PluginOsdComponent::image_url => $image_url,
			PluginOsdComponent::x => $x,
			PluginOsdComponent::y => $y,
			PluginOsdComponent::image_width => $image_width,
			PluginOsdComponent::image_height => $image_height
		);
	}

	public static function add_osd_text(&$comps, $text, $x, $y, $text_font_size = PLUGIN_FONT_NORMAL, $text_color = DEF_LABEL_TEXT_COLOR_WHITE, $text_halo = false)
	{
		if (!class_exists('PluginOsdComponent'))
			return;

		$comps[] = array
		(
			PluginOsdComponent::text => $text,
			PluginOsdComponent::x => $x,
			PluginOsdComponent::y => $y,
			PluginOsdComponent::text_font_size => $text_font_size,
			PluginOsdComponent::text_color => $text_color,
			PluginOsdComponent::text_halo => $text_halo
		);
	}

	public static function update_osd($comps, $post_action = null)
	{
		if (!defined('PLUGIN_UPDATE_OSD_ACTION_ID'))
			return $post_action;

		return
			array
			(
				 GuiAction::handler_string_id => PLUGIN_UPDATE_OSD_ACTION_ID,
				 GuiAction::data =>
					 array
					 (
						PluginUpdateOsdActionData::components => $comps,
						PluginUpdateOsdActionData::post_action => $post_action,
					 ),
			);
	}

    public static function switch_osd_mode($osd_mode = null, $post_action = null)
    {
    	if (!defined('SWITCH_OSD_MODE_ACTION_ID'))
    		return $post_action;

	    return
		    array
		    (
		         GuiAction::handler_string_id => SWITCH_OSD_MODE_ACTION_ID,
		         GuiAction::data =>
			         array
			         (
			            SwitchOsdModeActionData::osd_mode => $osd_mode,
			            SwitchOsdModeActionData::post_action => $post_action,
			         ),
		    );
    }

	public static function update_info_block($text_above, $text_color = null, $text_halo = false, $text_y_offset = 0, $post_action = null)
	{
		if (!defined('PLUGIN_UPDATE_INFO_BLOCK_ACTION_ID') || !class_exists('PluginUpdateInfoBlockActionData'))
			return $post_action;

		return
			array
			(
				GuiAction::handler_string_id => PLUGIN_UPDATE_INFO_BLOCK_ACTION_ID,
				GuiAction::data => array
				(
					PluginUpdateInfoBlockActionData::text_above => $text_above,
					PluginUpdateInfoBlockActionData::text_color => $text_color,
					PluginUpdateInfoBlockActionData::text_halo => $text_halo,
					PluginUpdateInfoBlockActionData::text_y_offset => $text_y_offset,
					PluginUpdateInfoBlockActionData::post_action => $post_action,
				 ),
			);
	}

    ///////////////////////////////////////////////////////////////////////////


    // Archives ///////////////////////////////////////////////////////////////

	public static function clear_archive_cache($archive_id = null, $post_action = null)
	{
		return
			parent::clear_archive_cache($archive_id, $post_action);
	}

	///////////////////////////////////////////////////////////////////////////


    // System /////////////////////////////////////////////////////////////////

	public static function change_settings($settings, $reboot, $restart_gui, $post_action = null)
	{
		if (!defined('CHANGE_SETTINGS_ACTION_ID') || !class_exists('ChangeSettingsActionData'))
			return $post_action;

		return
			array
			(
				 GuiAction::handler_string_id => CHANGE_SETTINGS_ACTION_ID,
				 GuiAction::data =>
					 array
					 (
						ChangeSettingsActionData::settings => $settings,
						ChangeSettingsActionData::reboot => $reboot,
						ChangeSettingsActionData::restart_gui => $restart_gui,
						ChangeSettingsActionData::post_action => $post_action,
					 ),
			);
	}

	public static function run_system_action($run_string, $post_action = null)
	{
		if (!defined('PLUGIN_SYSTEM_ACTION_ID') || !class_exists('PluginSystemActionData'))
			return $post_action;

		return
			array
			(
				GuiAction::handler_string_id => PLUGIN_SYSTEM_ACTION_ID,
				GuiAction::data =>
					array
					(
						PluginSystemActionData::run_string => $run_string,
						PluginSystemActionData::post_action => $post_action,
					),
			);
	}

    public static function run_builtin_action($builtin_action_id, $params = null, $post_action = null, $error_action = null)
    {
        if (!defined('RUN_BUILTIN_ACTION_ID') || !class_exists('RunBuiltinActionData'))
        	return isset($error_action)? $error_action : $post_action;

	    return
		    array
		    (
		        GuiAction::handler_string_id => RUN_BUILTIN_ACTION_ID,
		        GuiAction::data =>
			        array
			        (
			            RunBuiltinActionData::builtin_action_id => $builtin_action_id,
			            RunBuiltinActionData::params => $params,
			            RunBuiltinActionData::post_action => $post_action,
			            RunBuiltinActionData::error_action => $error_action,
			        ),
		    );
    }

	public static function show_main_screen($post_action = null)
	{
		return
			parent::show_main_screen($post_action);
	}

	public static function show_black_screen($bg_url, $post_action = null)
	{
		return
			array
			(
				 GuiAction::handler_string_id => SHOW_BLACK_SCREEN_ACTION_ID,
				 GuiAction::data =>
					 array
					 (
			 			ShowBlackScreenActionData::bg_url => $bg_url,
						ShowBlackScreenActionData::post_action => $post_action,
					 ),
			);
	}

	public static function replace_path($erase_count = null, $elements = null, $post_action = null)
	{
		if (!defined('PLUGIN_REPLACE_PATH_ACTION_ID') || !class_exists('PluginReplacePathActionData'))
			return $post_action;

		return
			array
			(
				GuiAction::handler_string_id => PLUGIN_REPLACE_PATH_ACTION_ID,
				GuiAction::data =>
					array
					(
						PluginReplacePathActionData::erase_count => $erase_count,
						PluginReplacePathActionData::elements => $elements,
						PluginReplacePathActionData::post_action => $post_action,
					),
			);
	}

	public static function rename_path($caption, $post_action = null)
	{
		if (!defined('RENAME_PATH_ACTION_ID') || !class_exists('RenamePathActionData'))
			return $post_action;

		return
			array
			(
				GuiAction::handler_string_id => RENAME_PATH_ACTION_ID,
				GuiAction::data =>
					array
					(
						RenamePathActionData::caption => $caption,
						RenamePathActionData::post_action => $post_action,
					),
			);
	}

    public static function add_menu_items($menu_items)
    {
    	if (!defined('ADD_MENU_ITEMS_ACTION_ID'))
    		return null;

        return
	        array
	        (
	            GuiAction::handler_string_id => ADD_MENU_ITEMS_ACTION_ID,
	            GuiAction::data =>
		            array
		            (
		                AddMenuItemsActionData::menu_items => $menu_items,
		            ),
	        );
    }

    public static function clear_watch_history($item_ids = null, $post_action = null)
    {
    	if (!defined('CLEAR_WATCH_HISTORY_ACTION_ID'))
    		return $post_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => CLEAR_WATCH_HISTORY_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                ClearWatchHistoryActionData::item_ids => $item_ids,
		                ClearWatchHistoryActionData::post_action => $post_action,
		             ),
	        );
    }

    public static function check_system_storage($required_space, $recommended_space, $title, $messages, $post_action = null)
    {
    	if (!defined('CHECK_SYSTEM_STORAGE_ACTION_ID'))
    		return $post_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => CHECK_SYSTEM_STORAGE_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                CheckSystemStorageActionData::required_space => $required_space,
		                CheckSystemStorageActionData::recommended_space => $recommended_space,
		                CheckSystemStorageActionData::messages => $messages,
		                CheckSystemStorageActionData::post_action => $post_action,
		             ),
	        );
    }

    public static function init_persistfs($post_action = null)
    {
    	if (!defined('INIT_PERSISTFS_ACTION_ID'))
    		return $post_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => INIT_PERSISTFS_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                CheckSystemStorageActionData::post_action => $post_action,
		             ),
	        );
    }

    public static function wait_ext_app_start($delay, $package, $error_action = null)
    {
    	if (!defined('WAIT_EXT_APP_START_ACTION_ID'))
    		return $error_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => WAIT_EXT_APP_START_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                WaitExtAppStartActionData::delay => $delay,
		                WaitExtAppStartActionData::package => $package,
		                WaitExtAppStartActionData::error_action => $error_action,
		             ),
	        );
    }

    public static function launch_ext_app($cmd, $ensure_playback_stopped, $delay, $package, $error_action = null)
    {
    	if (!defined('LAUNCH_EXT_APP_ACTION_ID'))
    		return $error_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => LAUNCH_EXT_APP_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                LaunchExtAppActionData::cmd => $cmd,
		                LaunchExtAppActionData::ensure_playback_stopped => $ensure_playback_stopped,
		                LaunchExtAppActionData::delay => $delay,
		                LaunchExtAppActionData::package => $package,
		                LaunchExtAppActionData::error_action => $error_action,
		             ),
	        );
    }

    public static function download_and_install_apk($caption, $package, $urls, $indirect = false, $post_action = null)
    {
    	if (!defined('DOWNLOAD_AND_INSTALL_APK_ACTION_ID'))
    		return $post_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => DOWNLOAD_AND_INSTALL_APK_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                DownloadAndInstallApkActionData::caption => $caption,
		                DownloadAndInstallApkActionData::package => $package,
		                DownloadAndInstallApkActionData::urls => $urls,
		                DownloadAndInstallApkActionData::indirect => $indirect,
		                DownloadAndInstallApkActionData::post_action => $post_action,
		             ),
	        );
    }

    public static function add_to_favorites($media_url, $caption = null, $icon_url = null, $skip_interactions = false, $post_action = null)
    {
    	if (!defined('ADD_TO_FAVORITES_ACTION_ID'))
    		return $post_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => ADD_TO_FAVORITES_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                AddToFavoritesActionData::media_url => $media_url,
		                AddToFavoritesActionData::caption => $caption,
		                AddToFavoritesActionData::icon_url => $icon_url,
		                AddToFavoritesActionData::skip_interactions => $skip_interactions,
		                AddToFavoritesActionData::post_action => $post_action,
		             ),
	        );
    }

    public static function remove_from_favorites($name, $post_action = null)
    {
    	if (!defined('REMOVE_FROM_FAVORITES_ACTION_ID'))
    		return $post_action;

        return
	        array
	        (
	             GuiAction::handler_string_id => REMOVE_FROM_FAVORITES_ACTION_ID,
	             GuiAction::data =>
		             array
		             (
		                RemoveFromFavoritesActionData::name => $name,
		                RemoveFromFavoritesActionData::post_action => $post_action,
		             ),
	        );
    }

	public static function stb_reboot()
	{
		if (defined('RESTART_ACTION_ID'))
			return
				array
				(
					GuiAction::handler_string_id => RESTART_ACTION_ID,
					GuiAction::data => array(
					RestartActionData::reboot => true),
				);

		if ($action = self::change_settings(null, true, false))
			return $action;

		exec('reboot');
	}

	public static function stb_shell_reboot()
	{
		if (defined('RESTART_ACTION_ID'))
			return
        		array
        		(
					GuiAction::handler_string_id => RESTART_ACTION_ID,
					GuiAction::data => array(RestartActionData::reboot => false),
				);

		if ($action = self::change_settings(null, false, true))
			return $action;

		exec('killall shell');
	}

	///////////////////////////////////////////////////////////////////////////

}

?>
